<?php
/*
Plugin Name: Bible Verse Content Generator
Description: Generate Bible verse content based on user input with server-side image checking and quota system (AJAX version)
Version: 2.3
Author: Jiten Panday
*/

// Include the feature image generator
require_once plugin_dir_path(__FILE__) . 'feature-image-generator.php';

function check_for_plugin_update() {
    $plugin_data = get_plugin_data(__FILE__);
    $current_version = $plugin_data['Version'];
    $plugin_slug = plugin_basename(__FILE__);
    
    $remote_version = wp_remote_get('https://bibletool.site/plugin/plugin-info.json');
    $remote_version = json_decode(wp_remote_retrieve_body($remote_version));
    
    if (version_compare($current_version, $remote_version->version, '<')) {
        $plugin_updates = get_site_transient('update_plugins');
        $update = new stdClass();
        $update->slug = $plugin_slug;
        $update->new_version = $remote_version->version;
        $update->url = $remote_version->download_url;
        $update->package = $remote_version->download_url;
        $plugin_updates->response[$plugin_slug] = $update;
        set_site_transient('update_plugins', $plugin_updates);
    }
}
add_action('admin_init', 'check_for_plugin_update');

// Add menu item
add_action('admin_menu', 'bvcg_add_menu');

function bvcg_add_menu() {
    add_menu_page('Bible Verse Content Generator', 'Bible Verse Generator', 'manage_options', 'bvcg-generator', 'bvcg_generator_page');
    add_submenu_page('bvcg-generator', 'Settings', 'Settings', 'manage_options', 'bvcg-settings', 'bvcg_settings_page');
}

// Generator page
function bvcg_generator_page() {
    $api_key = get_option('bvcg_api_key', '');
    $server_url = get_option('bvcg_server_url', '');
    ?>
    <div class="wrap">
        <h1>Bible Verse Content Generator</h1>
        <?php
        if (empty($api_key) || empty($server_url)) {
            echo '<div class="notice notice-warning"><p>Please configure your API key and server URL in the <a href="' . admin_url('admin.php?page=bvcg-settings') . '">settings page</a>.</p></div>';
        } else {
            $connection_status = bvcg_check_connection();
            if ($connection_status === true) {
                echo '<div class="notice notice-success"><p>Connected to server successfully!</p></div>';
                
                // Fetch additional information from the server
                $server_info = bvcg_get_server_info();
                if ($server_info !== false) {
                    echo '<div class="notice notice-info">';
                    echo '<ul>';
                    echo '<li>Name: '.'<strong>' . esc_html($server_info['server_name']) .'</strong>'. '</li>';
                    echo '<li>Price: ' . esc_html($server_info['version']) . '</li>';
                    echo '<li>Quota: ' . esc_html($server_info['last_updated']) . '</li>';
                    echo '</ul>';
                    echo '</div>';

                    if (isset($server_info['message'])) {
                        echo '<div class="notice notice-info">';
                        echo '<p><strong>Message:</strong></p>';
                        echo '<p>' . wp_kses_post($server_info['message']) . '</p>';
                        echo '</div>';
                    }
                }

                // Add a div for displaying quota information
                echo '<div id="bvcg-quota-info"></div>';
            } else {
                echo '<div class="notice notice-error"><p>Failed to connect to server: ' . esc_html($connection_status) . '</p></div>';
            }
        }

        // Add AJAX nonce
        wp_nonce_field('bvcg_generate_content', 'bvcg_nonce');
        ?>
        <style>
        .bvcg-form-container {
            max-width: 600px;
            margin: 20px auto;
            padding: 20px;
            background-color: #ffffff;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }

        .bvcg-form {
            display: flex;
            flex-direction: column;
        }

        .bvcg-input {
            padding: 12px;
            margin-bottom: 15px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 16px;
        }

        .bvcg-button {
            padding: 12px 20px;
            background-color: #4CAF50;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
            transition: background-color 0.3s ease;
        }

        .bvcg-button:hover {
            background-color: #45a049;
        }

        #bvcg-loading {
            display: none;
            text-align: center;
            margin-top: 20px;
        }

        #bvcg-result {
            margin-top: 20px;
        }

        @media (max-width: 768px) {
            .bvcg-form-container {
                max-width: 100%;
                margin: 10px;
            }
        }
        </style>

        <div class="bvcg-form-container">
            <form id="bvcg-form" class="bvcg-form">
                <input type="text" name="bvcg_keyword" id="bvcg_keyword" placeholder="Enter Bible Topic keyword" required class="bvcg-input">
                <input type="submit" value="Generate Content" class="bvcg-button">
            </form>
        </div>
        <div id="bvcg-loading">Generating content... Please wait.</div>
        <div id="bvcg-result"></div>
    </div>

    <script>
    jQuery(document).ready(function($) {
        function updateQuota() {
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'bvcg_update_quota_ajax',
                    nonce: $('#bvcg_nonce').val()
                },
                success: function(response) {
                    if (response.success) {
                        $('#bvcg-quota-info').html(response.data);
                    } else {
                        $('#bvcg-quota-info').html('<div class="notice notice-error"><p>Failed to fetch quota information.</p></div>');
                    }
                },
                error: function() {
                    $('#bvcg-quota-info').html('<div class="notice notice-error"><p>An error occurred while fetching quota information.</p></div>');
                }
            });
        }

        // Update quota on page load
        updateQuota();

        $('#bvcg-form').on('submit', function(e) {
            e.preventDefault();
            var keyword = $('#bvcg_keyword').val();
            var nonce = $('#bvcg_nonce').val();

            $('#bvcg-loading').show();
            $('#bvcg-result').empty();

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'bvcg_generate_content_ajax',
                    keyword: keyword,
                    nonce: nonce
                },
                success: function(response) {
                    $('#bvcg-loading').hide();
                    $('#bvcg-result').html(response.data);
                    // Update quota after content generation
                    updateQuota();
                },
                error: function() {
                    $('#bvcg-loading').hide();
                    $('#bvcg-result').html('<div class="notice notice-error"><p>An error occurred. Please try again.</p></div>');
                }
            });
        });
    });
    </script>
    <?php
}

// Settings page
function bvcg_settings_page() {
    if (isset($_POST['bvcg_save_settings'])) {
        update_option('bvcg_api_key', sanitize_text_field($_POST['bvcg_api_key']));
        update_option('bvcg_server_url', esc_url_raw($_POST['bvcg_server_url']));
        update_option('bvcg_default_category', intval($_POST['bvcg_default_category']));
        update_option('afi_auto_featured_image', sanitize_text_field($_POST['afi_auto_featured_image']));
        update_option('afi_text_color', sanitize_hex_color($_POST['afi_text_color']));
        echo '<div class="notice notice-success"><p>Settings saved successfully!</p></div>';
    }

    if (isset($_POST['bvcg_test_connection'])) {
        $connection_status = bvcg_check_connection();
        echo $connection_status === true 
            ? '<div class="notice notice-success"><p>Connection successful!</p></div>'
            : '<div class="notice notice-error"><p>Connection failed: ' . esc_html($connection_status) . '</p></div>';
    }

    $api_key = get_option('bvcg_api_key', '');
    $server_url = get_option('bvcg_server_url', '');
    $default_category = get_option('bvcg_default_category', '');
    $afi_auto_featured_image = get_option('afi_auto_featured_image', 'none');
    $afi_text_color = get_option('afi_text_color', '#0400b7');

    // Get all categories
    $categories = get_categories(array('hide_empty' => false));
    ?>
    <div class="wrap">
        <h1>Bible Verse Content Generator Settings</h1>
        <form method="post" action="">
        <h2>Plugin Settings Instructions</h2>
<p>Please follow the instructions below to configure the plugin settings:</p>
<ul>
    <li><strong>API Key and Server URL:</strong> Enter your API key and server URL. You can get a free demo API key and server URL by clicking <a href="https://bibletool.site/bible-plugin/" target="_blank">here</a>.</li> <hr>
    <li><strong>Default Category:</strong> Choose the category where the generated posts will be saved. You must select a category for your posts.</li><hr>
    <li><strong>Auto Featured Image:</strong> This feature generates a featured image automatically for your posts. 
        <ul>
            <li><strong>None:</strong> No featured image will be generated.</li>
            <li><strong>When Drafted:</strong> The featured image is generated when a post is drafted.</li>
            <li><strong>When Published:</strong> The featured image is generated when a post is published.</li>
            <li><strong>Both:</strong> The featured image is generated when a post is drafted and again when it is published. We recommend selecting<strong> "Both" </strong>for the best experience.</li><hr>
        </ul>
    </li>
    <li><strong>Featured Image Text Color:</strong> You can choose the color of the text that appears on the featured image. Click <a href="https://bibletool.site/bible-plugin/main_image/" target="_blank">here</a> to preview how the text will look on the image.</li>
</ul>

<table class="form-table">
    <tr>
        <th><label for="bvcg_api_key">API Key</label></th>
        <td><input type="text" id="bvcg_api_key" name="bvcg_api_key" value="<?php echo esc_attr($api_key); ?>" class="regular-text"></td>
    </tr>
    <tr>
        <th><label for="bvcg_server_url">Server URL</label></th>
        <td><input type="url" id="bvcg_server_url" name="bvcg_server_url" value="<?php echo esc_url($server_url); ?>" class="regular-text"></td>
    </tr>
    <tr>
        <th><label for="bvcg_default_category">Default Category</label></th>
        <td>
            <select id="bvcg_default_category" name="bvcg_default_category">
                <option value="">Select a category</option>
                <?php
                foreach ($categories as $category) {
                    echo '<option value="' . esc_attr($category->term_id) . '" ' . selected($default_category, $category->term_id, false) . '>' . esc_html($category->name) . '</option>';
                }
                ?>
            </select>
        </td>
    </tr>
    <tr>
        <th><label for="afi_auto_featured_image">Auto Featured Image</label></th>
        <td>
            <select name="afi_auto_featured_image" id="afi_auto_featured_image">
                <option value="none" <?php selected($afi_auto_featured_image, 'none'); ?>>None</option>
                <option value="draft" <?php selected($afi_auto_featured_image, 'draft'); ?>>When Drafted</option>
                <option value="publish" <?php selected($afi_auto_featured_image, 'publish'); ?>>When Published</option>
                <option value="both" <?php selected($afi_auto_featured_image, 'both'); ?>>Both</option>
            </select>
        </td>
    </tr>
    <tr>
        <th><label for="afi_text_color">Featured Image Text Color</label></th>
        <td>
            <input type="text" name="afi_text_color" id="afi_text_color" value="<?php echo esc_attr($afi_text_color); ?>" class="wp-color-picker-field" data-default-color="#0400b7" />
        </td>
    </tr>
</table>

            <p class="submit">
                <input type="submit" name="bvcg_save_settings" value="Save Settings" class="button button-primary">
                <input type="submit" name="bvcg_test_connection" value="Test Connection" class="button button-secondary">
            </p>
        </form>
    </div>
    <script>
        jQuery(document).ready(function($) {
            $('.wp-color-picker-field').wpColorPicker();
        });
    </script>
    <?php
}

// Check connection
function bvcg_check_connection() {
    $api_key = get_option('bvcg_api_key', '');
    $server_url = get_option('bvcg_server_url', '');

    if (empty($api_key) || empty($server_url)) {
        return 'API key or server URL is missing';
    }

    $response = wp_remote_post($server_url, array(
        'headers' => array('X-API-Key' => $api_key),
        'body' => array('action' => 'check_connection'),
        'timeout' => 5,
    ));

    if (is_wp_error($response)) {
        return $response->get_error_message();
    }

    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        return 'Server returned status code: ' . $response_code;
    }

    $body = wp_remote_retrieve_body($response);
    $content = json_decode($body, true);

    if (!$content || !isset($content['status']) || $content['status'] !== 'success') {
        return isset($content['error']) ? $content['error'] : 'Invalid response from server';
    }

    return true;
}

// Add AJAX action for content generation
add_action('wp_ajax_bvcg_generate_content_ajax', 'bvcg_generate_content_ajax');

function bvcg_generate_content_ajax() {
    check_ajax_referer('bvcg_generate_content', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized access');
    }

    $keyword = sanitize_text_field($_POST['keyword']);
    ob_start();
    bvcg_generate_content($keyword);
    $content = ob_get_clean();
    wp_send_json_success($content);
}

// Add AJAX action for updating quota
add_action('wp_ajax_bvcg_update_quota_ajax', 'bvcg_update_quota_ajax');

function bvcg_update_quota_ajax() {
    check_ajax_referer('bvcg_generate_content', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized access');
    }

    $quota_data = bvcg_get_quota();
    if ($quota_data !== false) {
        $remaining_quota = $quota_data['total_quota'] - $quota_data['used_quota'];
        if ($remaining_quota <= 0) {
            $html = '<div class="notice notice-error">
                <p>Your quota limit has been reached. To continue using the Bible Verse Content Generator, please contact the plugin owner at 
                <a href="mailto:jitenpanday3@gmail.com">jitenpanday3@gmail.com</a> to renew your quota.</p>
                <p>Once your quota is renewed, you will regain access to the search functionality.</p>
                </div>';
        } else {
            $html = '<div class="notice notice-info"><p>Remaining quota: ' . $remaining_quota . ' out of ' . $quota_data['total_quota'] . '</p></div>';
        }
        wp_send_json_success($html);
    } else {
        wp_send_json_error('Failed to fetch quota information');
    }
}

// Generate content
function bvcg_generate_content($keyword) {
    $api_key = get_option('bvcg_api_key', '');
    $server_url = get_option('bvcg_server_url', '');

    if (empty($api_key) || empty($server_url)) {
        echo '<div class="notice notice-error"><p>API key or server URL is missing. Please check your settings.</p></div>';
        return;
    }

    $response = wp_remote_post($server_url, array(
        'headers' => array('X-API-Key' => $api_key),
        'body' => array('action' => 'generate_content', 'idtest' => $keyword),
        'timeout' => 300,
    ));

    if (is_wp_error($response)) {
        echo '<div class="notice notice-error"><p>Error: ' . esc_html($response->get_error_message()) . '</p></div>';
        return;
    }

    $body = wp_remote_retrieve_body($response);
    $content = json_decode($body, true);

    if (!$content || $content['status'] !== 'success') {
        echo '<div class="notice notice-error"><p>Error: ' . (isset($content['error']) ? esc_html($content['error']) : 'Unknown error occurred') . '</p></div>';
        return;
    }

    // Get the default category
    $default_category = get_option('bvcg_default_category', '');
    // Define $main_title
    $main_title = $content['title']; // Main title is set here.

    // Process content directly without callbacks to avoid function scope issues.
   $content_with_images = $content['verses'];

    // Regex to match all blockquotes and insert images manually before them
    if (preg_match_all('/<blockquote class="wp-block-quote">(.+?)<\/blockquote>/is', $content_with_images, $matches)) {
        foreach ($matches[1] as $key => $blockquoteContent) {
            // Extract the Bible verse reference inside the blockquote
            if (preg_match('/<strong class="verse">— (.+?)<\/strong>/', $blockquoteContent, $verseMatch)) {
                $verseReference = trim($verseMatch[1]);

                if (!empty($verseReference)) {
                    // Format the title for image search
                    $formattedTitle = "Bible Verse (" . $verseReference . ")";
                    $customAltText = $main_title . " (" . $verseReference . ")";

                    // Query the database for an image with the alt text that matches $formattedTitle
                    $args = array(
                        'post_type'      => 'attachment',
                        'post_mime_type' => 'image',
                        'post_status'    => 'inherit',
                        'posts_per_page' => 1, // Limit to one image for performance reasons
                        'meta_query'     => array(
                            array(
                                'key'     => '_wp_attachment_image_alt',
                                'value'   => $formattedTitle, // This checks if the alt text contains the formatted title
                                'compare' => 'LIKE',
                            ),
                        ),
                    );

                    // Query the database for the image
                    $query = new WP_Query($args);
                    if ($query->have_posts()) {
                        $query->the_post(); // Get the first matched image
                        $image_url = wp_get_attachment_url(get_the_ID()); // Get image URL
                        
                        $image_html = '<figure class="wp-block-image size-full">';
                        // Create image HTML
                        $image_html .= '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($customAltText) . '">';
                        $image_html .= '</figure>';
                        // Insert the image before the blockquote
                        $content_with_images = str_replace($matches[0][$key], $image_html . $matches[0][$key], $content_with_images);
                    }

                    // Reset post data after the query
                    wp_reset_postdata();
                }
            }
        }
    }

    // Prepare post data
    $post_data = array(
        'post_title'    => $content['title'],
        'post_content'  => $content_with_images,
        'post_status'   => 'draft',
        'post_author'   => get_current_user_id(),
        'post_type'     => 'post',
        'post_category' => $default_category ? array($default_category) : array(),
    );

    // Insert the post
    $post_id = wp_insert_post($post_data);

    if ($post_id) {
        // Set featured image if enabled
        $afi_option = get_option('afi_auto_featured_image', 'none');
        if ($afi_option !== 'none') {
            $auto_featured_image = new Auto_Featured_Image();
            $auto_featured_image->afi_set_featured_image($post_id, get_post($post_id));
        }

        $category_name = '';
        if ($default_category) {
            $category = get_category($default_category);
            if ($category) {
                $category_name = $category->name;
            }
        }

        echo '<div class="notice notice-success"><p>Post created successfully' . ($category_name ? ' in category "' . esc_html($category_name) . '"' : '') . '. <a href="' . esc_url(get_edit_post_link($post_id)) . '">Edit post</a></p></div>';
        echo '<a href="' . get_permalink($post_id) . '" target="_blank">Post complete. Click here to view the post.</a>';
    } else {
        echo '<div class="notice notice-error"><p>Error creating post</p></div>';
    }
}

// Get quota from server
function bvcg_get_quota() {
    $api_key = get_option('bvcg_api_key', '');
    $server_url = get_option('bvcg_server_url', '');

    if (empty($api_key) || empty($server_url)) {
        return false;
    }

    $response = wp_remote_post($server_url, array(
        'headers' => array('X-API-Key' => $api_key),
        'body' => array('action' => 'get_total_quota'),
        'timeout' => 15,
    ));

    if (is_wp_error($response)) {
        error_log('Error fetching quota: ' . $response->get_error_message());
        return false;
    }

    $body = wp_remote_retrieve_body($response);
    $content = json_decode($body, true);

    if (isset($content['status']) && $content['status'] === 'success' && isset($content['total_quota']) && isset($content['used_quota'])) {
        return array(
            'total_quota' => $content['total_quota'],
            'used_quota' => $content['used_quota']
        );
    } else {
        error_log('Invalid response when fetching quota');
        return false;
    }
}

// Get server information
function bvcg_get_server_info() {
    $api_key = get_option('bvcg_api_key', '');
    $server_url = get_option('bvcg_server_url', '');

    if (empty($api_key) || empty($server_url)) {
        return false;
    }

    $response = wp_remote_post($server_url, array(
        'headers' => array('X-API-Key' => $api_key),
        'body' => array('action' => 'get_server_info'),
        'timeout' => 15,
    ));

    if (is_wp_error($response)) {
        error_log('Error fetching server info: ' . $response->get_error_message());
        return false;
    }

    $body = wp_remote_retrieve_body($response);
    $content = json_decode($body, true);

    if (isset($content['status']) && $content['status'] === 'success') {
        return $content['server_info'];
    } else {
        error_log('Invalid response when fetching server info');
        return false;
    }
}

// Initialize the Auto Featured Image functionality
new Auto_Featured_Image();